package net.haaks.spidertrap;

import java.util.ArrayList;

import org.newdawn.slick.Color;
import org.newdawn.slick.Graphics;
import org.newdawn.slick.SpriteSheet;
import org.newdawn.slick.geom.Circle;
import org.newdawn.slick.geom.Vector2f;
import org.newdawn.slick.util.Log;

public class Spider extends AbstractEntity {
	
	private static final int BOUNDSRADIUS = 14;
	
	private int direction = NODIR;
	private static SpriteSheet spiderMoveImages = null;
	private static SpriteSheet spiderStunnedImages = null;
	
	private int startIndex, endIndex, stunnedIndex, index, sheetrow;
	// indices are start,end,stunned. First index into array is direction
	// directions are: up=0, right=1, down=2, left=3
	// internal arrays mean row, startindex, endindex, stunnedindex
	private int[][] indices = {{1,0,10,1},{2,0,10,2},{3,0,10,3},{0,0,10,0}};
	
	private boolean stunned = false;
	
	private boolean moving = false;
	private int nextStep = 0;
	private final int NEXTSTEP = 20;	// in milliseconds
	private int SPEED = 2;
	
	private int id;
	private static int nextid = 1;
	
	private int hatchTimer = 0;
	private int nextHatchTime = 0;
	private static final int MINHATCHTIME = 30000;
	private static final int MAXHATCHTIME = 40000;
	
	private int stunnedTimer = 0;
	private static final int MINSTUNNEDTIME = 1000;	// spiders stay stunned for at least a second
	
	public static int MAXSPIDERS = 10;
	private static boolean breedingEnabled = true;
	private static boolean wakingUpStunnedOnes = true;
	
	public static Spider AddSpider(AreaMap map, Vector2f pos, CollisionGroup cg, GameContext context) {
//		Log.debug("Spider.AddSpider() is called with pos " + pos.toString());
		Spider spider;
		
		if (spiderMoveImages == null)
			spiderMoveImages = RessourceManager.getSpriteSheet("spider");
		if (spiderStunnedImages == null)
			spiderStunnedImages = RessourceManager.getSpriteSheet("spiderstunned");

		if (nrOfEggsAndSpiders() <= MAXSPIDERS) {
			spider = new Spider(pos);
			spider.gameContext = context;
			map.addEntity(spider, AreaMap.OBJECTLAYER);
			spider.cg = cg;
			if (cg != null)
				spider.cg.add(spider);
			spider.setDirection(spider.randomDirection());
			spider.nextHatchTime = spider.randomValue(MINHATCHTIME, MAXHATCHTIME);
			return spider;
		}
		return null;
	}
	
	public Spider(Vector2f startTile) {
		x = (int)startTile.x * SpiderTrap.TILESIZE;
		y = (int)startTile.y * SpiderTrap.TILESIZE;
		
		bounds = new Circle(x+SpiderTrap.TILESIZE_HALF, y+SpiderTrap.TILESIZE_HALF, BOUNDSRADIUS);
		offsetBoundsX = SpiderTrap.TILESIZE_HALF - BOUNDSRADIUS;
		offsetBoundsY = SpiderTrap.TILESIZE_HALF - BOUNDSRADIUS;
		moving = false;
		tile = startTile;
		toTile = null;
		
		id = nextid;
		nextid++;
		
	}

	public Circle getBounds() {
		return bounds;
	}

	public Entity getOwner() {
		return this;
	}

	public void setMap(AreaMap map) {
		this.map = map;
	}

	public void update(int delta) {
		if (!moving) {
			if (stunned) {
				// simply do nothing. maybe someone wakes us up
				stunnedTimer += delta;
				return;
			} else {
				// check for hatching
				hatchTimer += delta;
				if (hatchTimer > nextHatchTime) {
					hatchTimer = 0;
					nextHatchTime = randomValue(MINHATCHTIME, MAXHATCHTIME);
					if (nrOfEggsAndSpiders() < MAXSPIDERS && breedingEnabled) {
						// lay an egg at current spider position
						/*
						Log.debug("Spider.update(): Laying an egg at " + tile + ", size of cg is " + cg.getElements().size());
						Log.debug("cg is " + cg.toString());
						*/
						Egg.AddEgg(map, tile, CollisionGroup.GetGroup("eggs"), gameContext);
					}
				}
				// keep the current direction or select a new direction or get stunned if all directions are blocked
				// first check for valid directions
				// Log.debug("Spider.update(): tile is " + tile);
				ArrayList<Integer> dirs = availableDirections();
				nextStep = 0;
				if (dirs == null) {
					// we are stunned!
					moving = false;
					direction = NODIR;
					index = stunnedIndex;
					stunned = true;
					gameContext.propagate(GameContext.SPIDERSTUNNED, tile, null, null, null);
				} else {
					int oldDirection = direction;
					// keep the old direction if possible
					if (!dirs.contains(direction)) {
						// choose a random new one
						direction = randomDirectionFrom(dirs);
					} else {
						// add a bit of randomness. If random int(100) is > 95 then change direction
						if (randomChance(100,95)) {
							direction = randomDirectionFrom(dirs);
						}
					}
					if (direction != oldDirection)
						setDirection(direction);
					switch (direction) {
						case UP:
							toTile = new Vector2f(tile.x, tile.y - 1);
							dx = 0; dy = -SPEED; break;
						case DOWN:
							toTile = new Vector2f(tile.x, tile.y + 1);
							dx = 0; dy = SPEED; break;
						case RIGHT:
							toTile = new Vector2f(tile.x + 1, tile.y);
							dx = SPEED; dy = 0; break;
						case LEFT:
							toTile = new Vector2f(tile.x - 1, tile.y);
							dx = -SPEED; dy = 0; break;
					}
					moving = true;
				}
			}
		}
		if (moving) {
			hatchTimer += delta;
			// move til we fit the tile perfectly
			nextStep += delta;
			if (nextStep >= NEXTSTEP) {
				x += dx;
				y += dy;
				index ++;
				if (index > endIndex)
					index = startIndex;
				nextStep -= NEXTSTEP;
				if (x % SpiderTrap.TILESIZE == 0 && y % SpiderTrap.TILESIZE == 0) {
					// reached the next tile. stop moving.
					moving = false;
					tile = toTile; toTile = null;
					nextStep = 0;
				}
			}
			bounds.setX(x+offsetBoundsX);
			bounds.setY(y+offsetBoundsY);
			/*
			bounds.setCenterX(x + SpiderTrap.TILESIZE_HALF);
			bounds.setCenterY(y + SpiderTrap.TILESIZE_HALF);
			*/
//			Log.debug("x = " + x +", y = " + y + ", oBX = " + offsetBoundsX + ", oBY = " + offsetBoundsY +
//					", cx = " + bounds.getX() + ", cy = " + bounds.getY());
//			Log.debug("startindex = " + startIndex + ", endindex = " + endIndex + ", index = " + index );
		}
	}

	public void draw(Graphics g) {
		if (stunned)
			spiderStunnedImages.getSprite(index, 0).draw(x, y);
		else
			spiderMoveImages.getSprite(index, sheetrow).draw(x, y);
		/*
		*/
//		g.setColor(Color.red);
//		g.draw(bounds);
	}

	public int getDirection() {
		return direction;
	}

	public void setDirection(int direction) {
		this.direction = direction;
		this.sheetrow = indices[direction][0];
		this.startIndex = indices[direction][1];
		this.endIndex = indices[direction][2];
		this.stunnedIndex = indices[direction][3];
		this.index = this.startIndex;
	}
	
	private ArrayList<Integer> availableDirections() {
		// check tiles up, right, down and left
		Vector2f	testPos;
//		String[] collGroups = {"robots", "guards"};
		String[] collGroups = {"robots"};
		ArrayList<Integer>	res = new ArrayList<Integer>();
		
		testPos = new Vector2f(tile.x, tile.y-1);
		if (map.isPositionFree(collGroups, testPos))
			res.add(UP);
		testPos = new Vector2f(tile.x+1, tile.y);
		if (map.isPositionFree(collGroups, testPos))
			res.add(RIGHT);
		testPos = new Vector2f(tile.x, tile.y+1);
		if (map.isPositionFree(collGroups, testPos))
			res.add(DOWN);
		testPos = new Vector2f(tile.x-1, tile.y);
		if (map.isPositionFree(collGroups, testPos))
			res.add(LEFT);
		if (res.size() > 0)
			return res;
		return null;
	}

	public boolean isStunned() {
		return stunned;
	}

	public boolean hitBy(Entity source) {
		if (source instanceof Spider) {
			// wake up ourselves
			// Log.debug("spider with spider collision");
			if (wakingUpStunnedOnes) {
				if (stunnedTimer >= MINSTUNNEDTIME) {
					this.stunned = false;
					this.stunnedTimer = 0;
				}
			}
		/*
		} else if (source instanceof Bullet || source instanceof Guardian) {
			// bullets and guardians kill us
			gameContext.propagate(GameContext.SPIDERDESTROYED, tile, null, null, null);
			map.removeEntity(this);
			map.addExplosion(x + SpiderTrap.TILESIZE, y + SpiderTrap.TILESIZE);
			return true; // remove us from our collision group
		*/
		}
		return false;
		
	}
	
	public String toString() {
		return ("Spider id=" + id + ", tile=" + tile.toString() + ", stunned=" + stunned);
	}

	public static boolean isBreedingEnabled() {
		return breedingEnabled;
	}

	public static void setBreedingEnabled(boolean breedingEnabled) {
		Spider.breedingEnabled = breedingEnabled;
	}
	
	public static int nrOfEggsAndSpiders() {
		int eggs = 0, spiders = 0;
		
		spiders = CollisionGroup.GetGroup("spiders").getElements().size();
		eggs = CollisionGroup.GetGroup("eggs").getElements().size();
//		Log.debug("nrOfEggsAndSpiders(): spiders = " + spiders + ", eggs = " + eggs);
		return (spiders + eggs);
	}

	public static boolean isWakingUpStunnedOnes() {
		return wakingUpStunnedOnes;
	}

	public static void setWakingUpStunnedOnes(boolean wakingUpStunnedOnes) {
		Spider.wakingUpStunnedOnes = wakingUpStunnedOnes;
	}

}
