package net.haaks.spidertrap;

import org.newdawn.slick.Color;
import org.newdawn.slick.Font;
import org.newdawn.slick.GameContainer;
import org.newdawn.slick.Graphics;
import org.newdawn.slick.Input;
import org.newdawn.slick.Music;
import org.newdawn.slick.SlickException;
import org.newdawn.slick.SpriteSheet;
import org.newdawn.slick.geom.Vector2f;
import org.newdawn.slick.state.BasicGameState;
import org.newdawn.slick.state.StateBasedGame;
import org.newdawn.slick.state.transition.FadeInTransition;
import org.newdawn.slick.state.transition.FadeOutTransition;
import org.newdawn.slick.util.Log;

public class OptionsState extends BasicGameState implements GameContext {

	private Font bigFont, bigOrangeFont, smallFont;
	private TextScroller titleText = null, explanationText = null;
	
	private String[] title = { "Options" };
	
	private String[] options = {
			"Music volume",
			"SFX volume",
			"Skip tutorials"
	};
	private int[] values = new int[3];
	
	/** The index of the selected option */
	private int selected;
	
	private String[] explanation = {
			"Use Cursor up / Cursor down to switch options",
			"Use Cursor left / Cursor right to change values",
			"Press Return to save and back to main menu",
			"Press Escape for main menu"
	};
	
	/** The game holding this state */
	private StateBasedGame myGame;
	private GameContainer myContainer;

	// graphics and sound stuff
	private Music music = null;

	private AreaMap map;	// the map for the background
	
	private CollisionManager cgManager = CollisionManager.getSingleton();
	private CollisionGroup cgOptions = null;
	private boolean stateLeft = false;
	private SpriteSheet bigSpiderImages;
	private int index = 0;		// index to spider sprite
	private int nextstep = 0;
	private final int NEXTSTEP = 100;
	

	@Override
	public int getID() {
		return SpiderTrap.OPTIONS_STATE;
	}

	public void propagate(int notification, Vector2f tile, String infoString,
			Integer infoInt, Double infoDouble) {
		// nothing to propagate
	}

	public void init(GameContainer container, StateBasedGame game)
			throws SlickException {
		SpiderTrap.initRessources();
		
		myGame = game;
		myContainer = container;
		bigOrangeFont = RessourceManager.getAngelCodeFont("orangeinfo");
		bigFont = RessourceManager.getAngelCodeFont("info");
		smallFont = RessourceManager.getAngelCodeFont("hud");
		bigSpiderImages = RessourceManager.getSpriteSheet("bigspider");

		music = RessourceManager.getMusic("titlemusic");
		
		map = new AreaMap(this, "res/title.tmx");
		// center camera on map
		Vector2f center = new Vector2f(SpiderTrap.WIDTH_HALF, SpiderTrap.HEIGHT_HALF);
		map.setFocusObj(new SimpleMapFocus(center));
		map.setCameraStart(center);
		cgOptions = CollisionGroup.GetGroup("spiders");
	}

	public void render(GameContainer container, StateBasedGame game, Graphics g)
			throws SlickException {
		if (stateLeft )
			return;
		if (map != null) {
			map.draw(g);
		}
		if (titleText != null)
			titleText.render(g);
		if (explanationText != null)
			explanationText.render(g);

		g.setFont(bigFont);
		g.setColor(Color.white);
		for (int i=0;i<options.length;i++) {
			String text = options[i] + " - " + (values[i]);
			if (i == 2)
				text = options[i] + " - " + ((values[i] == 0) ? "off" : "on");
			if (selected == i) {
				bigSpiderImages.getSprite(index, 0).draw(SpiderTrap.WIDTH_HALF - 220, 120+(i*50));
				bigSpiderImages.getSprite(index, 1).draw(SpiderTrap.WIDTH_HALF + (220-64), 120+(i*50));
				g.drawString(text, SpiderTrap.WIDTH_HALF - (bigFont.getWidth(text)/2), 120+(i*50));
			} else {
				g.drawString(text, SpiderTrap.WIDTH_HALF - (bigFont.getWidth(text)/2), 120+(i*50));
			}
		}
	}

	public void update(GameContainer container, StateBasedGame game, int delta)
			throws SlickException {
		nextstep += delta;
		if (nextstep > NEXTSTEP) {
			nextstep -= NEXTSTEP;
			index++;
			if (index > 10)
				index = 0;
		}
//		RessourceManager.setMusicVolume((float)values[0]/(float)10);
//		RessourceManager.setSfxVolume((float)values[1]/(float)10);
		map.update(delta);
		cgManager.update();
		titleText.update(delta);
		explanationText.update(delta);
	}

	public void keyReleased(int key, char c) {
		if (key == Input.KEY_DOWN) {
			selected++;
			if (selected >= options.length) {
				selected = 0;
			}
		}
		if (key == Input.KEY_UP) {
			selected--;
			if (selected < 0) {
				selected = options.length - 1;
			}
		}
		if (key == Input.KEY_LEFT) {
			values[selected] -= 1;
			if (values[selected] < 0)
				values[selected] = 0;
			switch (selected) {
				case 0: RessourceManager.setMusicVolume((float)values[selected]/10.0f); break;
				case 1: RessourceManager.setSfxVolume((float)values[selected]/10.0f); break;
				case 2: GameOptions.get().setSkipTutorials((values[selected] == 0) ? false : true);
						Log.debug("Skipping tutorials is " + GameOptions.get().isSkipTutorials());
						break;
			}
		}
		if (key == Input.KEY_RIGHT) {
			values[selected] += 1;
			if (values[selected] > 10)
				values[selected] = 10;
			if (selected == 2) {
				// the tutorial option can only be 0 (don't skip tutorials) or 1 (skip the tutorials)
				if (values[selected] > 1)
					values[selected] = 1;
				
			}
			switch (selected) {
				case 0: RessourceManager.setMusicVolume((float)values[selected]/10.0f); break;
				case 1: RessourceManager.setSfxVolume((float)values[selected]/10.0f); break;
				case 2: GameOptions.get().setSkipTutorials((values[selected] == 0) ? false : true);
						Log.debug("Skipping tutorials is " + GameOptions.get().isSkipTutorials());
						break;
			}
		}
		if (key == Input.KEY_RETURN) {
			IngameState igs = (IngameState) myGame.getState(SpiderTrap.INGAME_STATE);
			igs.setSkipTutorialLevels(GameOptions.get().isSkipTutorials());
			GameOptions.get().setMusicVolume(RessourceManager.getMusicVolume());
			GameOptions.get().setSfxVolume(RessourceManager.getSfxVolume());
			boolean retval = GameOptions.get().save();
			if (!retval)
				Log.error("Saving game options failed!");
			// back to title menu
			myGame.enterState(SpiderTrap.TITLE_STATE, new FadeOutTransition(Color.black),
					new FadeInTransition(Color.black));
		}
		if (key == Input.KEY_ESCAPE) {
			// back to title menu
			myGame.enterState(SpiderTrap.TITLE_STATE, new FadeOutTransition(Color.black),
					new FadeInTransition(Color.black));
		}
	}

	
	public void enter(GameContainer container, StateBasedGame game) throws SlickException {
		Log.debug("OptionsState.enter() is called");

		map.removeAllSpidersAndEggs();
		cgManager.clearAllCollisionGroups();
		Vector2f freepos = map.getFreePosition();
		Spider.AddSpider(map, freepos, cgOptions, this);

		titleText = new TextScroller(title, bigOrangeFont, Color.white, TextScroller.FORMAT_CENTER,
				0, 10, TextScroller.MOVE_UP, 20, 2);
		explanationText = new TextScroller(explanation, smallFont, Color.white, TextScroller.FORMAT_CENTER,
				5, 320, TextScroller.MOVE_UP, 25, 2);
		values[0] = (int) (RessourceManager.getMusicVolume()* 10 + 0.5f);
		values[1] = (int) (RessourceManager.getSfxVolume() * 10 + 0.5f) ;
		values[2] = (GameOptions.get().isSkipTutorials()) ? 1 : 0;
		if (!music.playing())
			music.loop();
	}

	public void leave(GameContainer container, StateBasedGame game) throws SlickException {
		Log.debug("OptionsState.leave() is called");
		cgOptions.clear();
	}

}
