package net.haaks.spidertrap;

import java.util.ArrayList;
import java.util.Iterator;

import org.lwjgl.input.Keyboard;
import org.newdawn.slick.AngelCodeFont;
import org.newdawn.slick.Color;
import org.newdawn.slick.GameContainer;
import org.newdawn.slick.Graphics;
import org.newdawn.slick.Image;
import org.newdawn.slick.Input;
import org.newdawn.slick.Music;
import org.newdawn.slick.SlickException;
import org.newdawn.slick.Sound;
import org.newdawn.slick.geom.Vector2f;
import org.newdawn.slick.openal.SoundStore;
import org.newdawn.slick.state.BasicGameState;
import org.newdawn.slick.state.StateBasedGame;
import org.newdawn.slick.state.transition.FadeInTransition;
import org.newdawn.slick.state.transition.FadeOutTransition;
import org.newdawn.slick.util.Log;
import org.newdawn.slick.util.ResourceLoader;

public class IngameState extends BasicGameState implements GameContext {
	
	public static final int FIRST_NON_TUTORIAL_LEVEL = 5;

	public static final int NONE = -1;
	
	public static final int NORMAL = 0;
	public static final int HARD = 1;
	
	public static final int MAXROBOTS = 8;
	public static final int ROBOT_ENERGY = 15;
	public static final int MAX_ENERGY = 100;

	private static final int LEVEL_TIME = 120;	// 2 minutes
	
	private static final int INGAME = 0;
	private static final int PAUSED = 1;
	
	private boolean stateLeft = true;
	
	private int internalState = INGAME;			// internal ingame state

	private StateBasedGame myGame;
	
	private boolean skipTutorialLevels = false;
	
	private int level = 0;
	private int fileLevel = 0;	// index of loaded areamap, rotates if end of level files is reached and increases difficulty

	private int difficulty = NORMAL;
	
	private int score = 0;
	private int energy = 100;
	public int timeLeftWithShield = LEVEL_TIME;
	private int timeLeftNoShield = LEVEL_TIME;
	private boolean switchedShieldsOff = false;
	private int milliSecs = 0;
	
	// stuff for showing the game over explosion for the computers
	private boolean gameOver = false;
	private int gameOverTimer = 0;
	private static final int GAME_OVER_TIME = 3000;

	private AreaMap map;
	private Robot[] robots = new Robot[MAXROBOTS];
	
	private int activeRobot = NONE;
	private int nrOfActiveRobots = 0;
	
	private Vector2f start = null;
	
	// graphics and sound stuff
	private AngelCodeFont hudFont = null, redHudFont = null, bonusFont = null;
	private Image energyImg = null;
	private Sound nopeSound = null;
	private Sound bonusSound = null;
	private Music music = null;
	
	// stuff for collision
	private CollisionGroup cgRobots, cgSpiders;
	private CollisionManager cgManager = CollisionManager.getSingleton();


	@Override
	public int getID() {
		return SpiderTrap.INGAME_STATE;
	}

	public void init(GameContainer container, StateBasedGame game) throws SlickException {
		SpiderTrap.initRessources();	

		myGame = game;
		
		hudFont = RessourceManager.getAngelCodeFont("hud");
		redHudFont = RessourceManager.getAngelCodeFont("redhud");
		bonusFont = RessourceManager.getAngelCodeFont("info");
		
		energyImg = RessourceManager.getImage("energy");
		
		nopeSound = RessourceManager.getSound("nope");
		bonusSound = RessourceManager.getSound("bonus");
		music = RessourceManager.getMusic("ingamemusic");

		difficulty = NORMAL;
		
		cgRobots = CollisionGroup.GetGroup("robots");
		cgSpiders = CollisionGroup.GetGroup("spiders");
		
		internalState = INGAME;
		restart(null);
	}
	
	public void restart(SavedGameData savedGame) {
		for (int i = 0; i < MAXROBOTS; i++)
			robots[i] = null;
		nrOfActiveRobots = 0;
		if (savedGame == null) {
			score = 0;
			energy = 100;
			level = 0;
			fileLevel = 0;
			if (skipTutorialLevels) {
				level = FIRST_NON_TUTORIAL_LEVEL - 1;
				fileLevel = level;
			}
		} else {
			score = savedGame.getScore();
			energy = savedGame.getEnergy();
			level = savedGame.getLevel();
			fileLevel = savedGame.getFileLevel();
		}
	}
	
	public void nextLevel(GameContainer container, StateBasedGame game) throws SlickException {
		
		// calculate proper number of energy to save the proper data
		activeRobot = NONE;
		for (int i = 0; i < MAXROBOTS; i++) {
			if (robots[i] != null) {
				energy += ROBOT_ENERGY;
				robots[i] = null;
			}
		}
		nrOfActiveRobots = 0;
		if (energy > MAX_ENERGY)
			energy = MAX_ENERGY;
		
		// save current game state
		boolean ret = SavedGameData.save(score, energy, level, fileLevel, difficulty);
		if (!ret)
			Log.error("Saving game data failed - no game resume possible!");

		// reset collision groups
		cgManager.clearAllCollisionGroups();
		
		// per default egg laying and waking up stunned spiders is true
		Spider.setBreedingEnabled(true);
		Spider.setWakingUpStunnedOnes(true);
		
		level++;
		fileLevel++;
		String levelFilename = "res/level" + Utils.pad(fileLevel,3) + ".tmx";
		boolean levelExists = true;
		try {
			ResourceLoader.getResourceAsStream(levelFilename);
		} catch (RuntimeException e) {
			levelExists = false;
		}
		if (!levelExists) {
			// no more level files, restart at FIRST_NON_TUTORIAL_LEVEL and increase difficulty
			fileLevel = FIRST_NON_TUTORIAL_LEVEL;
			difficulty++;
		}
		
		timeLeftWithShield = LEVEL_TIME;
		timeLeftNoShield = LEVEL_TIME;
		switchedShieldsOff = false;
		milliSecs = 0;
		gameOverTimer = 0;
		gameOver = false;
		
		map = new AreaMap(this, "res/level" + Utils.pad(fileLevel,3) + ".tmx");
		this.start = map.getStartPos();
		// let the map start centered above the start position
		Vector2f pixelStart = new Vector2f(start.x * SpiderTrap.TILESIZE, start.y * SpiderTrap.TILESIZE);
		map.setFocusObj(new SimpleMapFocus(pixelStart));
		map.setCameraStart(pixelStart);
		addMapTitle(map.getTitle());
		// generate appropriate amount of spiders
		for (int i = 0; i < (1+difficulty); i++) {
			Vector2f freepos = map.getFreePosition();
			Spider.AddSpider(map, freepos, cgSpiders, this);
		}
		// generate bonus object, initially not visible
		Bonus.AddBonus(map, this, false);
		
	}

	public void render(GameContainer container, StateBasedGame game, Graphics g) throws SlickException {
		if (stateLeft)
			return;
		switch (internalState) {
		case INGAME:
			if (map != null) {
				map.draw(g);
			}
			// draw particle systems
			drawHUD(g);
			break;
		case PAUSED:
			g.setColor(Color.white);
			g.setFont(hudFont);
			g.drawString("Game is paused", SpiderTrap.WIDTH_HALF - (hudFont.getWidth("Game is paused")/2), 200);
			g.drawString("Press Space to continue", SpiderTrap.WIDTH_HALF - (hudFont.getWidth("Press Space to continue")/2), 400);
			break;
		}
	}

	private void drawHUD(Graphics g) {
		g.setFont(hudFont);
		g.setColor(Color.white);
		g.drawString("Score: " + Utils.pad(score,6), 0, 0);
		g.drawString("Energy:", 220, 0);
		
		g.drawRect(315, 7, 101, 19);
		// neu ab hier
		g.drawImage(energyImg, 315, 7);
		g.setColor(Color.black);
		g.fillRect(315+energy, 7, 100-energy, 18);
		// ende neu ab hier
//		g.setColor(Color.green);
//		g.fillRect(316, 8, energy, 18);
		g.setColor(Color.white);
		if (timeLeftWithShield != -1) {
			g.drawString("Time Left: " + Utils.timePad(timeLeftWithShield), 430, 0);
		} else {
			g.setFont(redHudFont);
			g.drawString("Time Left: " + Utils.timePad(timeLeftNoShield), 430, 0);
		}
	}

	public void update(GameContainer container, StateBasedGame game, int delta) throws SlickException {
		if (stateLeft)
			return;

		Input input = container.getInput();

		switch (internalState) {
		case INGAME:
			if (!gameOver) {
				if (energy < ROBOT_ENERGY && nrOfActiveRobots == 0) {
					// not enough energy to create a new robot
					gameOver = true;
				}
				if (energy < 0)
					gameOver = true;
				
				if (input.isKeyDown(Keyboard.KEY_ESCAPE))
					proceedToTitleState();
				if (input.isKeyDown(Keyboard.KEY_P))
					internalState = PAUSED;
				if (activeRobot != NONE) {
					int x = 0, y = 0;
					if (input.isKeyDown(Keyboard.KEY_LEFT)) {
						x -= 1;
					} else if (input.isKeyDown(Keyboard.KEY_RIGHT)) {
						x += 1;
					} else if (input.isKeyDown(Keyboard.KEY_UP)) {
						y -= 1;
					} else if (input.isKeyDown(Keyboard.KEY_DOWN)) {
						y += 1;
					}
					robots[activeRobot].applyDirection(x, y);
				}
				if (input.isKeyDown(Keyboard.KEY_N) && SpiderTrap.wizardMode)
					proceedToNextLevelState();
			}
			
			// update the map and all entities
			map.update(delta);
			// update collision manager
			cgManager.update();
			// update particle systems
			if (areAllSpidersStunned()) {
				proceedToNextLevelState();
			}
			// update the time
			milliSecs += delta;
			if (milliSecs > 1000) {
				milliSecs -= 1000;
				if (timeLeftWithShield > 0) {
					timeLeftWithShield --;
				} else {
					if (!switchedShieldsOff) {
						switchedShieldsOff = true;
						switchComputerShieldsOff();
					}
					timeLeftWithShield = -1;
					timeLeftNoShield --;
				}
				if (timeLeftNoShield <= 0) {
					// game lost. Computers explode
					Log.debug("Game lost. Time ran out!");
					bombAllComputers();
					gameOver = true;
				}
			}
			if (gameOver) {
				gameOverTimer += delta;
				if (gameOverTimer > GAME_OVER_TIME) {
					proceedToGameOverState();
				}
			}
			break;
		case PAUSED:
			if (input.isKeyDown(Keyboard.KEY_SPACE))
				internalState = INGAME;
			break;
		}
	}
	
	public void keyReleased(int key, char c) {
		switch (key) {
			case Keyboard.KEY_1: activateRobot(0); break;
			case Keyboard.KEY_2: activateRobot(1); break;
			case Keyboard.KEY_3: activateRobot(2); break;
			case Keyboard.KEY_4: activateRobot(3); break;
			case Keyboard.KEY_5: activateRobot(4); break;
			case Keyboard.KEY_6: activateRobot(5); break;
			case Keyboard.KEY_7: activateRobot(6); break;
			case Keyboard.KEY_8: activateRobot(7); break;
		}	
	}
	
	private void bombAllComputers() {
		CollisionGroup cg = CollisionGroup.GetGroup("computer");
		if (cg == null) return;
		for (Iterator iter = cg.getElements().iterator(); iter.hasNext();) {
			Computer compi = (Computer) iter.next();
			compi.goKaboom();
		}
	}

	private void switchComputerShieldsOff() {
		CollisionGroup cg = CollisionGroup.GetGroup("computer");
		if (cg == null) return;
		for (Iterator iter = cg.getElements().iterator(); iter.hasNext();) {
			Computer compi = (Computer) iter.next();
			compi.setShieldsDown();
		}
	}

	private boolean areAllSpidersStunned() {
		CollisionGroup cg = CollisionGroup.GetGroup("spiders");
		if (cg == null) return true;
		ArrayList elems = cg.getElements();
		Iterator iter = elems.iterator();
		while (iter.hasNext()) {
			Spider sp = (Spider) iter.next();
			if (!sp.isStunned())
				return false;
		}
		return true;
	}


	private void proceedToGameOverState() {
		GameOverState target = (GameOverState) myGame.getState(SpiderTrap.GAMEOVER_STATE);
		target.setEndgameData(new StateTransferData(score, 0, level, difficulty, false));
		myGame.enterState(SpiderTrap.GAMEOVER_STATE, new FadeOutTransition(Color.black), new FadeInTransition(Color.black));
	}
	
	private void proceedToNextLevelState() {
		int secondsLeft = 0;
		if (timeLeftWithShield > 0)
			secondsLeft += timeLeftWithShield;

		int timebonus = secondsLeft * 5;
		score += timebonus;
		NextLevelState target = (NextLevelState) myGame.getState(SpiderTrap.NEXTLEVEL_STATE);
		target.setTransferData(new StateTransferData(score, timebonus, level, difficulty, false));
		myGame.enterState(SpiderTrap.NEXTLEVEL_STATE, new FadeOutTransition(Color.black), new FadeInTransition(Color.black));
		
	}
	

	private void proceedToTitleState() {
		myGame.enterState(SpiderTrap.TITLE_STATE, new FadeOutTransition(Color.black), new FadeInTransition(Color.black));
	}
	

	private void activateRobot(int i) {
		// stop old robot moving and reset its active state
//		Log.debug("activateRobot(): i = " + i + ", activeRobot = " + activeRobot + "robots[i] = " + robots[i]);
		
		if (activeRobot != NONE) {
			robots[activeRobot].applyDirection(0, 0);
			robots[activeRobot].setTheActiveOne(false);
		}
		
		if (activeRobot == i) {
			robots[activeRobot].setTheActiveOne(true);	// stay active
			map.setFocusObj(robots[activeRobot]);
			return;
		}
		activeRobot = NONE;
		if (robots[i] == null) {
			// prepare a new robot if starttile is free and enough energy is left
			if (energy >= ROBOT_ENERGY) {
				Teleporter teleporter = map.getTeleporter();
				if (map.isPositionFree(start, teleporter)) {
					teleporter.openForRobot(i);
					map.setFocusObj(teleporter);
					return;
				} else {
//					Log.debug("startpos is not free!");
					if (!nopeSound.playing())
						nopeSound.play();
					return;
				}
			} else {
				Log.debug("out of energy!");
				if (!nopeSound.playing())
					nopeSound.play();
				return;
			}
		}
		activeRobot = i;
		robots[activeRobot].setTheActiveOne(true);	// become active
		map.setFocusObj(robots[activeRobot]);
	}

	public int getDifficulty() {
		return difficulty;
	}

	public void setDifficulty(int difficulty) {
		this.difficulty = difficulty;
	}

	public int getActiveRobot() {
		return activeRobot;
	}

	public void setActiveRobot(int activeRobot) {
		this.activeRobot = activeRobot;
	}

	public Robot getRobot(int id) {
		if (id < 0 || id >= MAXROBOTS)
			return null;
		return robots[id];
	}
	
	public Vector2f getStart() {
		return start;
	}

	public void setStart(Vector2f start) {
		this.start = start;
	}
	

	public void enter(GameContainer container, StateBasedGame game) throws SlickException {
		Log.debug("IngameState.enter() is called");
		if (!music.playing())
			music.loop();
		Log.debug("music volume is " + SoundStore.get().getMusicVolume());
		internalState = INGAME;
		
		nextLevel(container, game);
		Log.debug("IngameState.enter() is finished");
		stateLeft = false;
	}

	public void leave(GameContainer container, StateBasedGame game) throws SlickException {
		Log.debug("IngameState.leave() is called");
		stateLeft = true;
//		music.stop();
	}

	public void propagate(int notification, Vector2f tile, String infoString, Integer infoInt, Double infoDouble) {
		switch (notification) {
		case GameContext.TELEPORTER_OPENED:
			robots[infoInt] = new Robot(map, start, infoInt+1, cgRobots, this, hudFont);
			energy -= ROBOT_ENERGY;
			nrOfActiveRobots++;
			map.addTeleportEffect();
			activeRobot = infoInt;
			robots[activeRobot].setTheActiveOne(true);	// become active
			map.setFocusObj(robots[activeRobot]);
			break;
		case GameContext.SPIDERSTUNNED:
			score += 100;
			bonusSound.play();
			addBonusText(tile, "100");
			break;
		case GameContext.SPIDERDESTROYED:
			score += 250;
			bonusSound.play();
			addBonusText(tile, "250");
			break;
		case GameContext.EGGDESTROYED:
			score += 50;
			bonusSound.play();
			addBonusText(tile, "50");
			break;
		case GameContext.ROBOTDESTROYED:
			// destroy the robot in our internal array to revive him later
			Log.debug("Robot destroyed, infoInt = " + infoInt + ", activeRobot = " + activeRobot);
			robots[infoInt] = null;
			nrOfActiveRobots--;
			if (activeRobot == infoInt)
				activeRobot = NONE;
			break;
		case GameContext.ANYTHINGHITCOMPUTER:
			// create explosions on all computers, wait a few seconds and game over
			bombAllComputers();
			map.setFocusObj(new SimpleMapFocus(tile));
			gameOver = true;
			break;
		case GameContext.ENERGYSUCKED:
			// player looses some energy
			energy -= 1;
			break;
		case GameContext.BONUSPICKEDUP:
			// player picks up a bonus
			if (infoInt == Bonus.TIME) {
				if (timeLeftWithShield > 0)
					timeLeftWithShield += 15;
				else
					timeLeftNoShield += 15;
				addBonusText(tile, "+15 seconds");
			} else if (infoInt == Bonus.ENERGY) {
				energy += 10;
				addBonusText(tile, "+10 energy");
				if (energy > 100)
					energy = 100;
			}
		}
	}
	
	private void addBonusText(Vector2f tile, String text) {
		this.map.addFloatingText(text, bonusFont, (int)tile.x * SpiderTrap.TILESIZE,
				(int)tile.y * SpiderTrap.TILESIZE, 4, 100, 5000, 0.02f);
	}
	

	private void addMapTitle(String title) {
		// let the map title appear centered above the start position
		int tw2 = bonusFont.getWidth(title) / 2;
		Vector2f pixelStart = new Vector2f(start.x * SpiderTrap.TILESIZE, start.y * SpiderTrap.TILESIZE);
		this.map.addFloatingText(title, bonusFont, (int) pixelStart.x - tw2,
				(int) pixelStart.y, 2, 100, 5000, 0.01f);
//		Log.debug("Adding map title '" + title + "'");
	}

	public boolean isSkipTutorialLevels() {
		return skipTutorialLevels;
	}

	public void setSkipTutorialLevels(boolean skipTutorialLevels) {
		this.skipTutorialLevels = skipTutorialLevels;
	}

}
