package net.haaks.spidertrap;

import org.newdawn.slick.Color;
import org.newdawn.slick.Font;
import org.newdawn.slick.GameContainer;
import org.newdawn.slick.Graphics;
import org.newdawn.slick.Input;
import org.newdawn.slick.Music;
import org.newdawn.slick.SlickException;
import org.newdawn.slick.geom.Vector2f;
import org.newdawn.slick.gui.AbstractComponent;
//import org.newdawn.slick.gui.BasicComponent;
import org.newdawn.slick.gui.ComponentListener;
import org.newdawn.slick.gui.TextField;
import org.newdawn.slick.state.BasicGameState;
import org.newdawn.slick.state.StateBasedGame;
import org.newdawn.slick.state.transition.FadeInTransition;
import org.newdawn.slick.state.transition.FadeOutTransition;
import org.newdawn.slick.util.Log;

public class GameOverState extends BasicGameState implements GameContext, ComponentListener {

	
	private StateBasedGame myGame = null;
	private StateTransferData data = null;
	
	private boolean stateLeft = true;

	private String[] gameOverText;
	private TextScroller infoText, highscoreText = null;
	private Font infoFont, highscoreFont;
	private Color infoColor = Color.white;
	
	// graphics and sound stuff
	private Music music = null;

	private AreaMap map;	// the map for the background
	
	private CollisionManager cgManager = CollisionManager.getSingleton();
	private CollisionGroup cgGameOver = null;

	// highscore stuff
	private HighscoreList highscores;
	private boolean enterHighscore = false;
	private TextField yourName;
	
	private String spaceText = "Press Space To Continue";
	private String enterNameText = "Please Enter Your Name:";
	
	@Override
	public int getID() {
		return SpiderTrap.GAMEOVER_STATE;
	}

	public void init(GameContainer container, StateBasedGame game) throws SlickException {
		SpiderTrap.initRessources();
		
		myGame = game;
		infoFont = RessourceManager.getAngelCodeFont("orangeinfo");
		highscoreFont = RessourceManager.getAngelCodeFont("info");
		highscores = new HighscoreList();

		music = RessourceManager.getMusic("titlemusic");
		
		map = new AreaMap(this, "res/gameover.tmx");
		// center camera on map
		Vector2f center = new Vector2f(SpiderTrap.WIDTH_HALF, SpiderTrap.HEIGHT_HALF);
		map.setFocusObj(new SimpleMapFocus(center));
		map.setCameraStart(center);
		cgGameOver = CollisionGroup.GetGroup("spiders");

	}

	public void render(GameContainer container, StateBasedGame game, Graphics g) throws SlickException {
		if (stateLeft)
			return;
		if (map != null) {
			map.draw(g);
		}
		if (infoText != null)
			infoText.render(g);
		if (highscoreText != null) {
			highscoreText.render(g);
			g.setColor(Color.white);
			g.drawString(spaceText, SpiderTrap.WIDTH_HALF - (infoFont.getWidth(spaceText)/2), 
					SpiderTrap.HEIGHT - (infoFont.getHeight("ABC")+20));
		}
		if (enterHighscore) {
			g.setColor(Color.white);
			g.drawString(enterNameText, SpiderTrap.WIDTH_HALF - (infoFont.getWidth(enterNameText)/2), 260);
			yourName.render(container, g);
		}

	}

	public void update(GameContainer container, StateBasedGame game, int delta) throws SlickException {
		if (stateLeft)
			return;
		map.update(delta);
		cgManager.update();
		if (infoText != null)
			infoText.update(delta);
		if (highscoreText != null)
			highscoreText.update(delta);
	}

	public void enter(GameContainer container, StateBasedGame game) throws SlickException {
		Log.debug("GameOverState.enter() is called");

		// delete existing saved games
		boolean ret = SavedGameData.clear();
		if (!ret)
			Log.error("Clearing game data failed - game resume still possible!");

		Vector2f freepos = map.getFreePosition();
		map.removeAllSpidersAndEggs();
		cgManager.clearAllCollisionGroups();
		Spider.AddSpider(map, freepos, cgGameOver, this);

		if (data == null)
			throw new SlickException("GameOverState.enter(): data is null!");
		highscoreText = null;
		if (highscores.isHighscore(data.getScore()) && !data.isShowOnly()) {
			int height = infoFont.getHeight("ABC") + 20;
			yourName = new TextField(container, infoFont, SpiderTrap.WIDTH_HALF - 100,
					300,200,height, this);
			yourName.setFocus(true);
			yourName.setMaxLength(10);
			String[] text = {
					"You Reached Level " + data.getLevel(),
					"With A Score Of " + data.getScore(),
					"You Made A Highscore!"
			};
			gameOverText = text;
			enterHighscore = true;
		} else {
			String[] text = {
					"You Reached Level " + data.getLevel(),
					"With A Score Of " + data.getScore()
			};
			String[] showOnlyText = { "The Highscores" };
			if (data.isShowOnly())
				gameOverText = showOnlyText;
			else
				gameOverText = text;
			
			highscoreText = this.generateHighscoreScroller();
		}
		infoText = new TextScroller(gameOverText, infoFont, infoColor, TextScroller.FORMAT_CENTER,
				0, 20, TextScroller.RandomMove(), 25, 2);
		stateLeft = false;
		if (!music.playing())
			music.loop();
	}

	public void leave(GameContainer container, StateBasedGame game) throws SlickException {
		Log.debug("GameOverState.leave() is called");
		stateLeft = true;
		cgGameOver.clear();
	}

	public void setEndgameData(StateTransferData data) {
		this.data = data;
	}

	public void keyReleased(int key, char c) {
		if (!enterHighscore) {
			if (key == Input.KEY_ESCAPE) {
				// back to title menu
				myGame.enterState(SpiderTrap.TITLE_STATE, new FadeOutTransition(Color.black), new FadeInTransition(Color.black));
			}
			if (key == Input.KEY_SPACE) {
				// back to title menu
				myGame.enterState(SpiderTrap.TITLE_STATE, new FadeOutTransition(Color.black), new FadeInTransition(Color.black));
			}
		}
	}

	public void componentActivated(AbstractComponent source) {
//	public void componentActivated(BasicComponent source) {
		// assure that we enter our name only once, even if componentActivated is called more often
		if (source == yourName && enterHighscore) {
			// now we can store the new highscore
			highscores.insertHighscore(data.getScore(), yourName.getText());
			enterHighscore = false;
			highscoreText = this.generateHighscoreScroller();
		}
	}

	public void propagate(int notification, Vector2f tile, String infoString, Integer infoInt, Double infoDouble) {
		// nothing to do, we ignore every propagation
		
	}
	
	private TextScroller generateHighscoreScroller() {
		TextScroller scr;
		int len = 6;
		String[] scoresStrings = new String[HighscoreList.MAX_ENTRIES];
		for (int i = 0; i < HighscoreList.MAX_ENTRIES; i++) {
			scoresStrings[i] = Utils.pad(highscores.getHighscoreAt(i), len) + "  " + highscores.getNameAt(i);
		}
		scr = new TextScroller(scoresStrings, highscoreFont, infoColor, TextScroller.FORMAT_CENTER,
				0, 160, TextScroller.RandomMove(), 25, 2);
		return scr;
	}

}
