package net.haaks.spidertrap;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Random;

import org.newdawn.slick.AngelCodeFont;
import org.newdawn.slick.Graphics;
import org.newdawn.slick.Image;
import org.newdawn.slick.SlickException;
import org.newdawn.slick.geom.Vector2f;
import org.newdawn.slick.particles.ConfigurableEmitter;
import org.newdawn.slick.particles.ParticleIO;
import org.newdawn.slick.particles.ParticleSystem;
import org.newdawn.slick.tiled.TiledMap;
import org.newdawn.slick.util.Log;

/**
 * @author Tommy Haaks
 */
public class AreaMap extends TiledMap {
	
	public static final int GROUNDLAYER = 0;
	public static final int OBJECTLAYER = 1;
	public static final int TOPLAYER = 2;
	
	/** the smooth camera speed, multiplied by delta later on */
	public static final float CAMERA_SPEED = 1.5f;
	
	/** ground indicates if there is a ground tile on these coordinates, blocked tells if it is free */
	private boolean ground[][], blocked[][];
	
	private String title = null;
	
	private GameContext context = null;

	//private ArrayList entities = new ArrayList();
	private ArrayList<Entity> groundEntities = new ArrayList<Entity>();
	private ArrayList<Entity> objectEntities = new ArrayList<Entity>();
	private ArrayList<Entity> topEntities = new ArrayList<Entity>();
	
	private ArrayList<Entity> removeMe = new ArrayList<Entity>();
	
	private FloatingTextBonusEmitter ftbEmitter = new FloatingTextBonusEmitter();
	
	private Vector2f startPos = new Vector2f(0,0);
	private float startPosX, startPosY;
	private Teleporter teleporter = null;	// the start teleporter object
	
	/** the object the areamap focuses on */
	private AreaMapFocus focusObj = null;
	
	/** the real camera center position */
	private int realCX = 0, realCY = 0;
	/** the maximum speed of the camera */
	private int cameraDelta = 0;	
	
	private ParticleSystem explosionSystem;
	private ParticleSystem bulletSystem;
	private BulletEmitter bulletEmitter;
//	private ParticleSystem splatSystem;
//	private SplatEmitter splatEmitter;
	private ParticleSystem effectSystem;
	private ConfigurableEmitter teleportEmitter, sparkEmitter, smallExplosionEmitter;
	
	// stuff for map attributes to control
	private String mapMessages[];
	private int msgTimer = 0, msgIndex = 0, lastMsgIndex = 0;
	private AngelCodeFont msgFont;
	private static final int MESSAGE_NEXT = 6000;
	
	public AreaMap(GameContext context, String ref) throws SlickException {
		super(ref);
		
		title = getMapProperty("title", "Another laboratory");
//		Log.debug("AreaMap: title is '" + title + "'");
		
		// check map messages
		String msgs = getMapProperty("messages", "");
		
		// check egg laying
		String eggs = getMapProperty("eggs", "true");
		if (eggs.equalsIgnoreCase("false")) {
			Spider.setBreedingEnabled(false);
			if (msgs.length() > 0)
				msgs += "|";
			msgs += "No eggs on this level.";
		} else
			Spider.setBreedingEnabled(true);
		// check waking up stunned spiders
		String wakeup = getMapProperty("wakeup", "true");
		if (wakeup.equalsIgnoreCase("false")) {
			Spider.setWakingUpStunnedOnes(false);
			if (msgs.length() > 0)
				msgs += "|";
			msgs += "Spiders stay stunned on this level.";
		} else
			Spider.setWakingUpStunnedOnes(true);
		
		// prepare array of msg strings
		if (msgs != null) {
			Log.debug("msgs is '" + msgs + "'");
			mapMessages = msgs.split("\\|");
//			Log.debug(mapMessages.toString());
			lastMsgIndex = mapMessages.length-1;
		} else
			mapMessages = null;

		this.context = context;
		
		try {
			msgFont = RessourceManager.getAngelCodeFont("info");
			explosionSystem = ParticleIO.loadConfiguredSystem("res/explosion.xml");
			ConfigurableEmitter emitter = (ConfigurableEmitter) explosionSystem.getEmitter(0);
			emitter.setEnabled(false);
			effectSystem = new ParticleSystem("res/particle.tga");
			effectSystem.setBlendingMode(ParticleSystem.BLEND_ADDITIVE);
			teleportEmitter = ParticleIO.loadEmitter("res/teleportemitter.xml");
			teleportEmitter.setEnabled(false);
			sparkEmitter = ParticleIO.loadEmitter("res/bluesparksemitter.xml");
			sparkEmitter.setEnabled(false);
			smallExplosionEmitter = ParticleIO.loadEmitter("res/smallexplosionemitter.xml");
			smallExplosionEmitter.setEnabled(false);
//			splatSystem = new ParticleSystem("res/particle.tga");
//			splatEmitter = new SplatEmitter();
//			splatSystem.addEmitter(splatEmitter);
			bulletSystem = new ParticleSystem(new Image("res/particle.tga"),500);
			bulletSystem.setBlendingMode(ParticleSystem.BLEND_ADDITIVE);
			bulletEmitter = new BulletEmitter();
			bulletSystem.addEmitter(bulletEmitter);
		} catch (IOException e) {
			throw new SlickException("AreaMap.AreaMap(): Failed to load particle systems", e);
		}

		// blocked information is in level 0
		ground = new boolean[width][height];
		blocked = new boolean[width][height];
		int layer = 0;
		for (int x = 0; x < width; x++) {
			for (int y = 0; y < height; y++) {
				int id = getTileId(x, y, layer);
				if (id != 0) {
					ground[x][y] = true;
					String prop = getTileProperty(id, "blocked", "false");
					if (prop.equals("true")) {
						blocked[x][y] = true;
					}
				}
			}
		}

		// object information is in level 1, level 2 only contains ceiling tiles
		layer = 1;
		for (int x = 0; x < width; x++) {
			for (int y = 0; y < height; y++) {
				int id = getTileId(x, y, layer);

				if (id != 0) {
					ground[x][y] = true;
					// there can also be blocking tiles on object layer (tables, cupboards)
					String prop = getTileProperty(id, "blocked", "false");
					if (prop.equals("true")) {
						blocked[x][y] = true;
					}
					String e = getTileProperty(id, "entity", "");
					if (!e.equals("")) {
						// here creation of start and computer objects
						if (e.equals("start")) {
							// startPos is the tile position
							startPos = new Vector2f(x,y);
							startPosX = x * SpiderTrap.TILESIZE + (SpiderTrap.TILESIZE / 2);
							startPosY = y * SpiderTrap.TILESIZE + (SpiderTrap.TILESIZE / 2);
							teleporter = Teleporter.AddTeleporter(this, startPos, context);
							// remove the placeholder graphics
							this.setTileId(x, y, layer, 0);
						}
						if (e.equals("computer")) {
							// add a computer entity later to draw shield and register collision...
							blocked[x][y] = true; // balls and spiders can't pass computers
							Vector2f tile = new Vector2f(x,y);
							ConfigurableEmitter emitter = this.addExplosionEmitter(tile);
							Computer.AddComputer(this, tile, emitter,
									CollisionGroup.GetGroup("computer"), context);
						}
						if (e.equals("turret")) {
							// add a turret in the proper direction
							String dir = getTileProperty(id, "direction", "");
							int direction = AbstractEntity.DirectionFromString(dir);
							Vector2f tile = new Vector2f(x,y);
							blocked[x][y] = true;
							switch (direction) {
								case AbstractEntity.UP: blocked[x][y+1] = true; break;
								case AbstractEntity.DOWN: blocked[x][y+1] = true; break;
								case AbstractEntity.LEFT: blocked[x+1][y] = true; break;
								case AbstractEntity.RIGHT: blocked[x+1][y] = true; break;
							}
							Turret.AddTurret(this, tile, direction, null, context);
							// remove the placeholder graphics
							this.setTileId(x, y, layer, 0);
						}
						if (e.equals("guardian")) {
							// add a guardian in the proper direction, either left or right
							String dir = getTileProperty(id, "direction", "LEFT");
							int direction = AbstractEntity.DirectionFromString(dir);
							Vector2f tile = new Vector2f(x,y);
							Guardian.AddGuardian(this, tile, direction, context);
							// remove the placeholder graphics
							this.setTileId(x, y, layer, 0);
						}
						if (e.equals("sucker")) {
							// add a energy sucker in the proper direction, either horizontal or vertical
							String dir = getTileProperty(id, "direction", "horizontal");
							int direction = AbstractEntity.LEFT;
							if (dir.equalsIgnoreCase("vertical"))
								direction = AbstractEntity.DOWN;
							Vector2f tile = new Vector2f(x,y);
							EnergySucker.AddEnergySucker(this, tile, direction, context);
							// remove the placeholder graphics
							this.setTileId(x, y, layer, 0);
						}
						// Log.debug("x=" + x + ", y=" + y + ", layer =" + layer + ", entity=" + e);
					}
				}
			}
		}
		// set camera to 0,0
		realCX = 0; realCY = 0;
	}
	
	private ConfigurableEmitter addExplosionEmitter(Vector2f tile) {
		if (explosionSystem != null) {
			ConfigurableEmitter newOne = null;
			ConfigurableEmitter explEmitter = (ConfigurableEmitter) explosionSystem.getEmitter(0);
			newOne = explEmitter.duplicate();
			if (newOne == null)
				return null;
			newOne.name = newOne.name + "_" + (int)tile.x + "_" + (int)tile.y;
			newOne.setPosition((int)tile.x * SpiderTrap.TILESIZE + SpiderTrap.TILESIZE/2,
					(int)tile.y * SpiderTrap.TILESIZE + SpiderTrap.TILESIZE/2);
			newOne.setEnabled(false);
			explosionSystem.addEmitter(newOne);
			return newOne;
		}
		return null;
	}

	public Vector2f getFreePosition() {
		Vector2f pos = null;
		Random rndGen = new Random();
		boolean found = false;
		while (!found) {
			int x = rndGen.nextInt(width);
			int y = rndGen.nextInt(height);
			pos = new Vector2f(x,y);
			if (isPositionFree(pos, null) && !pos.equals(startPos))
				found = true;
		}
		// loop will only terminate when we found a valid pos
		return pos;
	}
	

	public boolean isPositionFree(Vector2f tile, Entity against) {
		if (!isGround(tile))
			return false;
		
		if (isBlocked(tile)) {
			// Log.debug("isBlocked is true for tile " + tile.toString());
			return false;
		}
		if (against != null && isBlockedByEntity(against, tile)) {
			// Log.debug("isBlockedByEntity is true for tile " + tile.toString());
			return false;
		}
		return true;
	}
	
	private boolean isGround(Vector2f tile) {
		if (tile == null)
			return false;
		int x = (int) tile.x;
		int y = (int) tile.y;

		if ((x < 0) || (x >= width) || (y < 0) || (y >= height)) {
			return false;
		}
		return ground[x][y];
	}

	public boolean isPositionFree(String[] collisionGroups, Vector2f tile) {
		if (isBlocked(tile)) {
			// Log.debug("isBlocked is true for tile " + tile.toString());
			return false;
		}
		if (collisionGroups != null) {
			for (int i = 0; i < collisionGroups.length; i++) {
				CollisionGroup cg = CollisionGroup.GetGroup(collisionGroups[i]);
				if (cg != null) {
					ArrayList elems = cg.getElements();
					Iterator iter = elems.iterator();
					while (iter.hasNext()) {
						Entity en = (Entity) iter.next();
						if (en.isBlocked(tile))
							return false;
					}
				} else {
					// huh? collisiongroup does not exist
					Log.debug("AreaMap.isPositionFree(): There is no collision group named " + collisionGroups[i]);
				}
			}
		}
		return true;
	}


	public void setBlocked(int x, int y, boolean value) {
		if ((x < 0) || (x >= width) || (y < 0) || (y >= height)) {
			return;
		}
		blocked[x][y] = value;
		
	}
	
	public boolean isBlocked(int x, int y) {
		x /= SpiderTrap.TILESIZE;
		y /= SpiderTrap.TILESIZE;

		if ((x < 0) || (x >= width) || (y < 0) || (y >= height)) {
			return true;
		}

		return blocked[x][y];
	}
	
	public boolean isBlocked(Vector2f tile) {
		if (tile == null)
			return false;
		int x = (int) tile.x;
		int y = (int) tile.y;

		if ((x < 0) || (x >= width) || (y < 0) || (y >= height)) {
			return true;
		}

		return blocked[x][y];
	}
	
	// check for tile collisions by entities
	public boolean isBlockedByEntity(Entity against, Vector2f tile) {
		ArrayList entities = this.getEntities();
		
		for (int i = 0; i < entities.size(); i++) {
			Entity current = (Entity) entities.get(i);			
					
			if (against == null || ((current != against) && 
					(current.getOwner() != against.getOwner()) && 
					(against.getOwner() != current.getOwner()))) {
				if (current.isBlocked(tile)) {
					if (against != null)
						against.hitBy(current);
					return true;
				}
			}
		}
		return false;
	}

	
	public void addEntity(Entity entity, int layer) {
		entity.setMap(this);
		switch (layer) {
		case GROUNDLAYER: groundEntities.add(entity); break;
		case OBJECTLAYER: objectEntities.add(entity); break;
		case TOPLAYER: topEntities.add(entity); break;
		default: Log.error("AreaMap.addEntity(): illegal layer used!");
		}
	}

	public void removeEntity(Entity entity) {
		removeMe.add(entity);
	}

	public void draw(Graphics g) {
		int targetCX = 0;
		int targetCY = 0;

		if (focusObj != null) {
			// set center of areamap to focus obj
			targetCX = (int) focusObj.getPosition().x;
			targetCY = (int) focusObj.getPosition().y;
		} else {
			// set center of areamap to start position
			targetCX = (int) startPos.x * SpiderTrap.TILESIZE;
			targetCY = (int) startPos.y * SpiderTrap.TILESIZE;
		}
		// now smoothly move camera on realCX, realCY to targetCX, targetCY, using cameraDelta
		if (Math.abs(targetCX-realCX) > cameraDelta) {
			if (targetCX > realCX)
				realCX += cameraDelta;
			else
				realCX -= cameraDelta;
		} else
			realCX = targetCX;
		if (Math.abs(targetCY-realCY) > cameraDelta) {
			if (targetCY > realCY)
				realCY += cameraDelta;
			else
				realCY -= cameraDelta;
		} else
			realCY = targetCY;
		
		int sx = (realCX - SpiderTrap.WIDTH_HALF) / SpiderTrap.TILESIZE;
		int sy = (realCY - SpiderTrap.HEIGHT_HALF) / SpiderTrap.TILESIZE;
		int ox = (realCX - SpiderTrap.WIDTH_HALF) % SpiderTrap.TILESIZE;
		int oy = (realCY - SpiderTrap.HEIGHT_HALF) % SpiderTrap.TILESIZE;

		int x = -ox;
		int y = -oy;
		int height = SpiderTrap.HEIGHT / SpiderTrap.TILESIZE;
//		int width = SpiderTrap.WIDTH / SpiderTrap.TILESIZE;
		int width = (SpiderTrap.WIDTH / SpiderTrap.TILESIZE) + 1;

		// draw ground layer and entities
		drawLayer(GROUNDLAYER, x, y, sx, sy, realCX, realCY, width, g, groundEntities);
		// draw splatters below all objects
		g.translate(-realCX + SpiderTrap.WIDTH_HALF, -realCY + SpiderTrap.HEIGHT_HALF);
//		splatSystem.render();
		g.resetTransform();
		// draw object layer and entities
		drawLayer(OBJECTLAYER, x, y, sx, sy, realCX, realCY, width, g, objectEntities);
		g.translate(-realCX + SpiderTrap.WIDTH_HALF, -realCY + SpiderTrap.HEIGHT_HALF);
		bulletSystem.render();
		explosionSystem.render();
		effectSystem.render();
		g.resetTransform();
		// draw top layer and entities
		drawLayer(TOPLAYER, x, y, sx, sy, realCX, realCY, width, g, topEntities);
		g.translate(-realCX + SpiderTrap.WIDTH_HALF, -realCY + SpiderTrap.HEIGHT_HALF);
		ftbEmitter.renderSystem(g);
		g.resetTransform();
	}
	
	private void drawLayer(int layerIndex, int x, int y, int sx, int sy, int cx, int cy, int width, Graphics g, ArrayList<Entity> entities) {
		// draw the layer
		Layer layer = (Layer) layers.get(layerIndex);
		for (int ty = 0; ty < height; ty++) {
//			layer.render(x, y, sx, sy, width, ty, true, tileWidth, tileHeight);
			layer.render(x, y, sx, sy, width, ty, true);
		}
		// draw the entities on this layer
		for (Entity entity : entities) {
			float xp = -cx + SpiderTrap.WIDTH_HALF + entity.getX();
			float yp = -cy + SpiderTrap.HEIGHT_HALF + entity.getY();
			
			if ((xp > -50) && (yp > -50) && (xp < SpiderTrap.WIDTH + 50) && (yp < SpiderTrap.HEIGHT + 50)) {
				g.translate(-cx + SpiderTrap.WIDTH_HALF, -cy + SpiderTrap.HEIGHT_HALF);
				entity.draw(g);
				g.resetTransform();
			}
		}
		
	}

	public void update(int delta) {
		// first update particle systems
//		splatSystem.update(delta);
		bulletSystem.update(delta);
		effectSystem.update(delta);
		explosionSystem.update(delta);
		ftbEmitter.updateSystem(delta);
		ArrayList<Entity> entities = this.getEntities();
		for (Entity entity : entities) {
			entity.update(delta);
		}
		
		groundEntities.removeAll(removeMe);
		objectEntities.removeAll(removeMe);
		topEntities.removeAll(removeMe);
		removeMe.clear();
		// calculate cameraDelta based on delta
		cameraDelta = (int) (CAMERA_SPEED * delta);
		// update map messages
		if (mapMessages != null) {
			msgTimer += delta;
			if (msgTimer > MESSAGE_NEXT) {
				msgTimer -= MESSAGE_NEXT;
				addMapMessage(mapMessages[msgIndex]);
				msgIndex++;
				if (msgIndex > lastMsgIndex)
					mapMessages = null;
			}
		}
	}
	
	private void addMapMessage(String msg) {
		// let the map message appear centered on the screen
		int tw2 = msgFont.getWidth(msg) / 2;
		Vector2f pixelStart = new Vector2f(realCX, realCY);
		this.addFloatingText(msg, msgFont, (int) pixelStart.x - tw2,
				(int) pixelStart.y, 2, 100, 5000, 0.01f);
		
	}

	public Vector2f getStartPos() {
		return startPos;
	}

	public void setStartPos(Vector2f startPos) {
		this.startPos = startPos;
	}

	public ArrayList<Entity> getEntities() {
		ArrayList<Entity> entities = new ArrayList<Entity>();
		entities.addAll(groundEntities);
		entities.addAll(objectEntities);
		entities.addAll(topEntities);
		return entities;
	}
	
	public void addFloatingText(String text, AngelCodeFont font, int x, int y, int speedY, int speedDelta, int maxAliveMS, float alphaDec) {
		FloatingTextBonus bonus = new FloatingTextBonus(ftbEmitter, text, font, x, y, speedY, speedDelta, maxAliveMS, alphaDec);
		this.ftbEmitter.add(bonus);
	}
	
	public void addEggSplash(Vector2f tile) {
//		float x = tile.x * SpiderTrap.TILESIZE + SpiderTrap.TILESIZE_HALF;
//		float y = tile.y * SpiderTrap.TILESIZE + SpiderTrap.TILESIZE_HALF;
		EggSplat.AddEggSplat(this, tile, context);
//		splatEmitter.addSplat(x,y);
	}
	
	public void addBulletParticle(float x, float y) {
		bulletEmitter.add(x, y);
	}
	
	public void addTeleportEffect() {
//		Log.debug("AreaMap.addTeleportEffect() is called");
		ConfigurableEmitter e = teleportEmitter.duplicate();
		e.setEnabled(true);
		e.setPosition(startPosX, startPosY);
		effectSystem.addEmitter(e);
	}
	
	public void addExplosion(float x, float y) {
//		Log.debug("AreaMap.addExplosion() is called, x=" + x + ", y=" + y);
		ConfigurableEmitter e = smallExplosionEmitter.duplicate();
		e.setEnabled(true);
		e.setPosition(x, y);
		effectSystem.addEmitter(e);
	}
	
	public void addSparks(float x, float y) {
//		Log.debug("AreaMap.addSparks() is called, x=" + x + ", y=" + y);
		ConfigurableEmitter e = sparkEmitter.duplicate();
		e.setEnabled(true);
		e.setPosition(x, y);
		effectSystem.addEmitter(e);
	}
	
	public void removeAllSpidersAndEggs() {
		for (Entity entity : groundEntities) {
			if (entity instanceof Spider || entity instanceof Egg) {
//				Log.debug("removeAllSpidersAndEggs: remove " + entity.toString());
				this.removeEntity(entity);
			}
		}
		for (Entity entity : objectEntities) {
			if (entity instanceof Spider || entity instanceof Egg) {
//				Log.debug("removeAllSpidersAndEggs: remove " + entity.toString());
				this.removeEntity(entity);
			}
		}
	}

	public String getTitle() {
		return title;
	}

	public AreaMapFocus getFocusObj() {
		return focusObj;
	}

	public void setFocusObj(AreaMapFocus focusObj) {
		this.focusObj = focusObj;
	}
	
	public void setCameraStart(Vector2f start) {
		realCX = (int) start.x;
		realCY = (int) start.y;
	}

	public Teleporter getTeleporter() {
		return teleporter;
	}

}
